#!/bin/bash


set -e

ARGS=`getopt -o '' -l device:,out:,ftp:,http: -n "$0" -- "$@"`


if [ $? != 0 ]; then
    exit 1
fi
eval set -- "${ARGS}"

echo $@

DEVICE='gpu'
OUTPUT='detection-config.yml'
FTP_SERVER='ftp.server'
HTTP_DOMAIN='detection'

while true
do
    case "$1" in
    --device)
        DEVICE=$2
        echo Using $DEVICE
        shift 2
        ;;
    --out)
        OUTPUT=$2
        echo Output to $OUTPUT
        shift 2
        ;;
    --ftp)
        FTP_SERVER=$2
        echo Using FTP server $FTP_SERVER
        shift 2
        ;;
    --http)
        HTTP_DOMAIN=$2
        echo Using HTTP DOMAIN $HTTP_DOMAIN
        shift 2
        ;;
    --)
        echo Cli. parameters parsing done
        shift
        break
        ;;
    *)
        echo Wrong args $@
        exit 1
        ;;
    esac
done

DEVICE=`echo $DEVICE | tr a-z A-Z`

case "$DEVICE" in
CPU)
    DETECT_DEVICE_SELECT='cpu'
    ;;
GPU)
    DETECT_DEVICE_SELECT='0'
    ;;
*)
    echo Unknown device $DEVICE
    exit 1
    ;;
esac

echo Deleting old Filesystem
rm -f $OUTPUT

echo Writing namespace configuration
echo -e '\napiVersion: v1\nkind: Namespace\nmetadata:\n  name: detection\n' >> $OUTPUT

echo -e '\n---\n' >> $OUTPUT

echo Writing PV and PVC configuration
echo -e '\n# 创建一个 PV\napiVersion: v1\nkind: PersistentVolume\nmetadata:\n  name: infer-worker\n  labels:\n    type: local\n    app: infer-worker\n  namespace: detection\nspec:\n  storageClassName: local-path\n  capacity:\n    storage: 10Gi\n  volumeMode: Filesystem\n  accessModes:\n    - ReadWriteMany\n  hostPath:\n    # 指定了本机对应的存储位置\n    path: /data/infer-worker/model-weights\n\n---\n\n# 创建了一个 PVC\napiVersion: v1\nkind: PersistentVolumeClaim\nmetadata:\n  name: infer-worker\n  labels:\n    app: infer-worker\n  namespace: detection\nspec:\n  storageClassName: local-path\n  accessModes:\n    - ReadWriteMany\n  resources:\n    requests:\n      storage: 10Gi\n' >> $OUTPUT

echo -e '\n---\n' >> $OUTPUT

echo Writing ConfigMaps
echo -e '\napiVersion: v1\nkind: ConfigMap\nmetadata:\n  name: http-server\n  labels:\n    app: http-server\n  namespace: detection\ndata:\n  WORKER_URL: tcp://infer-worker.detection.svc.cluster.local:5001\n---\n\n# 创建了一个 ConfigMap （infer-worker)\napiVersion: v1\nkind: ConfigMap\nmetadata:\n  name: infer-worker\n  labels:\n    app: infer-worker\n  namespace: detection\ndata:\n  # 指定环境变量\n  # 指定 FTP 连接 IP（域名）\n  FTP_CONNECT_IP: '$FTP_SERVER'\n  # 指定 权重文件位置（容器中的路径）\n  MODEL_WEIGHTS_PATH: /app/best.pt\n  # 指定 CPU 或者 GPU\n  DETECT_DEVICE_SELECT: '$DETECT_DEVICE_SELECT'\n' >> $OUTPUT

echo -e '\n---\n' >> $OUTPUT

echo Writing Deployment
echo -e '\n# Deployment http-server\napiVersion: apps/v1\nkind: Deployment\nmetadata:\n  name: http-server\n  namespace: detection\nspec:\n  selector:\n    matchLabels:\n      app: http-server\n  replicas: 1\n  strategy:\n    type: Recreate\n  template:\n    metadata:\n      labels:\n        app: http-server\n    spec:\n      containers:\n      - name: http-server\n        image: http-server\n        args:\n        - "--bind=0.0.0.0:8080"\n        - "--timeout=1000"\n        imagePullPolicy: "Never"\n        envFrom:\n          - configMapRef:\n              name: http-server\n        ports:\n        - containerPort: 8080\n\n---\n\n# Deployment infer-worker\napiVersion: apps/v1\nkind: Deployment\nmetadata:\n  name: infer-worker\n  namespace: detection\nspec:\n  selector:\n    matchLabels:\n      app: infer-worker\n  replicas: 1\n  strategy:\n    type: Recreate\n  template:\n    metadata:\n      labels:\n        app: infer-worker\n    spec:\n      containers:\n      - name: infer-worker\n        image: infer-worker\n        imagePullPolicy: "Never"\n' >> $OUTPUT

if [ x"$DEVICE" == "XGPU" ]; then
    echo -e '        resources:\n          limits:\n            nvidia.com/gpu: 1\n' >> $OUTPUT
fi

echo -e '        envFrom: # 设置环境变量\n          - configMapRef:\n              name: infer-worker\n        ports:\n        - containerPort: 5001\n        volumeMounts:  # 将 上面创建的 PVC 绑定并挂载到容器的 /model 文件夹中\n          - mountPath: /model\n            name: model\n      volumes: # 声明的挂载的卷，采用上面创建的 PVC\n        - name: model\n          persistentVolumeClaim:\n            claimName: infer-worker\n' >> $OUTPUT

echo -e '\n---\n' >> $OUTPUT

echo Writing Service
echo -e '\n# Service\napiVersion: v1\nkind: Service\nmetadata:\n  name: http-server\n  labels:\n    app: http-server\n  namespace: detection\nspec:\n  selector:\n    app: http-server\n  ports:\n  - port: 8080\n    targetPort: 8080\n\n---\n\n# Service\napiVersion: v1\nkind: Service\nmetadata:\n  name: infer-worker\n  labels:\n    app: infer-worker\n  namespace: detection\nspec:\n  selector:\n    app: infer-worker\n  ports:\n  - port: 5001\n    targetPort: 5001\n' >> $OUTPUT

echo -e '\n---\n' >> $OUTPUT

echo Writing Ingress
echo -e '\n# ingress\n\napiVersion: networking.k8s.io/v1\nkind: Ingress\nmetadata:\n  name: proxy\n  namespace: detection\nspec:\n  rules:\n  - host: '$HTTP_DOMAIN'\n    http:\n      paths:\n      - path: /detect\n        pathType: Prefix\n        backend:\n          service:\n            name: http-server\n            port:\n              number: 8080\n' >> $OUTPUT


echo Done